/**
 * \file main.cpp
 * \author cfallin
 * \date 2008-01-23
 *
 * Main driver.
 */

#include "mem.h"
#include "cpu.h"
#include "mon.h"

#include <iostream>
#include <fstream>
#include <string>
#include <sstream>

#include <ncurses.h>

using namespace std;

string disasm(word w1, word w2, bool *twoword)
{
  ostringstream os;

  *twoword = false;
  const char *ops[32] =
    { "trap", "trap", "trap", "trap", "skz", "skn", "jr", "il",
      "add", "sub", "and", "or", "ld", "st", "ldim", "ldpc",
      "srl", "sll", "xor", "not", "mov", "movb", "mova", "high",
      "addi", "subi", "andi", "ori", "finish", "NOP", "NOP", "NOP" };

  int op = (w1 >> 10) & 63;
  const char *sop;
  if(op & 32)
    sop = "jmp";
  else
    sop = ops[op];

  os << sop << ' ';

  if(!(op & 32) && op >= 4)
  {
    int ifield = (w1 >> 8) & 3;
    if(ifield == 1)
      os << "rel ";
    else if(ifield == 2)
      os << "inc ";
    else if(ifield == 3)
      os << "dec ";
  }

  if(op & 32)
    os << "0x" << hex << setfill('0') << setw(4) << (w1 & 0x7fff);
  else
  {
    int ra = (w1 >> 4) & 15;
    int rb = w1 & 15;

    if(op >= 4)
      os << 'R' << ra << ' ';
    if(op == 8 || op == 9 || op == 10 || op == 11 || op == 12 || op == 13 ||
       op == 18 || op == 20 || op == 21 || op == 22)
      os << 'R' << rb;
    if(op == 24 || op == 25 || op == 26 || op == 27)
      os << rb;
    
    if(op == 14)
    {
      os << "0x" << hex << setfill('0') << setw(4) << (w2 & 0xffff);
      *twoword = true;
    }
  }

  return os.str();
}

void drawScreen(CPU *cpu, Memory *mem, Monitor *mon)
{
  clear();
  mvprintw(23, 0, "<ENTER> = step | q = quit | c = command");

  // regs at 0, 0
  string r = mon->doCommand("reg");
  mvprintw(0, 0, "%s", r.c_str());

  // disassembly around PC

  int disasm_start = cpu->PC();
  if(disasm_start < 10)
    disasm_start = 0;
  else
    disasm_start -= 10;

  int line = 0;
  for(int i = disasm_start; line < 20 && i < Memory::Words; i++, line++)
  {
    word w1 = mem->Read(i), w2 = mem->Read(i + 1);
    bool twoword = false;
    string insn = disasm(w1, w2, &twoword);
    char buf[256];

    if(twoword)
      snprintf(buf, 256, "%04x %04x", w1, w2);
    else
      snprintf(buf, 256, "%04x     ", w1);

    mvprintw(line, 30, "%c %04x | %s | %s",
             (i == cpu->PC()) ? '>' : ' ',
             i, buf, insn.c_str());

    if(twoword) i++;
  }

  move(22, 0);
}

void doMachine(WINDOW *w)
{
  // Construct the machine
  Memory mem;
  CPU cpu(&mem);
  Monitor mon(&cpu, &mem);

  // do it
  while(1)
  {
    drawScreen(&cpu, &mem, &mon);
    refresh();
    int ch = getch();
    if(ch == 'q') break;

    switch(ch)
    {
    case '\n':
      cpu.Step(1);
      break;
    case 'c':
      mvprintw(20,0,"Enter cmd: ");
      char b[256];
      echo();
      nocbreak();
      getnstr(b, 256);
      noecho();
      cbreak();

      string cmd(b);
      string ret = mon.doCommand(b);
      mvprintw(22,0,"%s\n[any key to continue]", ret.c_str());

      getch();
      break;
    }
  }
}

int main(int argc, char **argv)
{
  initscr();
  cbreak();
  noecho();

  doMachine(stdscr);

  endwin();
  return 0;
}
