/**
 * \file mon.cpp
 * \author cfallin
 * \date 2008-01-23
 *
 * Monitor (command language).
 */

#include "mon.h"

#include <string>
#include <sstream>
#include <iomanip>

using namespace std;

string Monitor::doCommand(string command)
{
  if(command == "")
    command = "step"; // shortcut

  // split off first word
  string cmdWord, cmdRest;
  int firstSpace = command.find(' ', 0);
  if(firstSpace == -1)
  {
    cmdWord = command;
    cmdRest = "";
  }
  else
  {
    cmdWord = command.substr(0, firstSpace);
    cmdRest = command.substr(firstSpace);
  }

  // dispatch the command
  if(cmdWord == "help")
    return doHelp(cmdRest);
  if(cmdWord == "load")
    return doLoad(cmdRest);
  if(cmdWord == "save")
    return doSave(cmdRest);
  if(cmdWord == "peek")
    return doPeek(cmdRest);
  if(cmdWord == "reg")
    return doReg(cmdRest);
  if(cmdWord == "reset")
    return doReset(cmdRest);
  if(cmdWord == "step")
    return doStep(cmdRest);
  if(cmdWord == "run")
    return doRun(cmdRest);
  if(cmdWord == "break")
    return doBreak(cmdRest);
  if(cmdWord == "count")
    return doCount(cmdRest);

  return "Unknown command. Enter 'help' to see a list of valid commands.";
}

string Monitor::doHelp(string args)
{
  return "\n"
    "  Commands:\n"
    "    help               - show this message\n"
    "    load <file>        - load memory from <file>, hex words\n"
    "    save <file>        - save memory to <file>, hex words\n"
    "    peek <addr>        - examine memory at <addr> (hex)\n"
    "    reg                - dump registers\n"
    "    reset              - reset CPU\n"
    "    step               - step one instruction\n"
    "    run                - run until breakpoint\n"
    "    break <addr>       - set the code breakpoint at <addr> (hex)\n"
    "                         (-1 to cancel)\n"
    "    <eof> (Ctrl-D)     - exit\n";
}

string Monitor::doLoad(string args)
{
  const char *p = args.c_str();

  // strip the filename
  while(*p && *p == ' ') p++;
  string filename(p);

  if(!m_mem->Load(filename))
    return "Could not load file into memory.";
  else
    return "";
}

string Monitor::doSave(string args)
{
  const char *p = args.c_str();

  // strip the filename
  while(*p && *p == ' ') p++;
  string filename(p);

  // save
  if(!m_mem->Save(filename))
    return "Could not save memory to file.";
  else
    return "";
}

string Monitor::doPeek(string args)
{
  // read the address
  istringstream is(args);
  int addr;
  is >> addr;

  word w = m_mem->Read(addr);

  ostringstream os;
  os << "0x" <<  setw(Memory::Width/4) << setfill('0') << hex << w;
  return os.str();
}

string Monitor::doReg(string args)
{
  ostringstream os;

  os << "PC: 0x" << setw(Memory::Width/4) << setfill('0') << hex <<
    m_cpu->m_pc << endl;
  os << "I: " << m_cpu->m_i << "; L: " << m_cpu->m_l << endl;
  for(int i = 0; i < CPU::NReg; i++)
    os << "R" << dec << setw(2) << i << ": 0x" <<  setw(Memory::Width/4) << setfill('0') << hex
       << m_cpu->m_regs[i] << endl;

  return os.str();
}

string Monitor::doReset(string args)
{
  m_cpu->Reset();
  return "";
}

string Monitor::doStep(string args)
{
  int x;
  if(args.length() > 0){
  x = atoi(args.c_str());
  m_cpu->Step(x);
  }
  else m_cpu->Step();
  return "";
}

string Monitor::doRun(string args)
{
  while(1)
  {
    if(!m_cpu->Step(1))
    {
      string s("Hit 'break' instruction.\n");
      return s;
    }

    if(m_cpu->m_pc == m_breakpoint)
    {
      string s("Hit breakpoint\n");
      return s;
    }
  }
}

string Monitor::doBreak(string args)
{
  // read the address
  istringstream is(args);
  int addr;
  is >> addr;

  m_breakpoint = addr;

  if(addr == -1)
  {
    string s("Canceled breakpoint.\n");
    return s;
  }
  else
  {
    string s("Set breakpoint\n");
    return s;
  }
}

string Monitor::doCount(string args)
{
  ostringstream os;

  os << "insns: " << m_cpu->m_count_insn << endl <<
        "jmps: " << m_cpu->m_count_jmp << endl;

  return os.str();
}
