/**
 * \file cpu.cpp
 * \author cfallin
 * \date 2008-01-23
 *
 * Emulator CPU core.
 */

#include "cpu.h"
#include <iostream>
#include <string.h> // memset

/* ------------ 12-bit arithmetic ------- */

// 12-to-full sign extend
#define SGNE12(x) ( ((x) & 0x800) ? - ((x) & 0x7ff) : ((x) & 0x7ff) )
// full-to-12 sign de-extend
#define SGND12(x) ( ((int)(x) < 0) ? (0x800 | ((x) & 0x7ff)) : ((x) & 0x7ff) )

/* ------------ 16-bit arithmetic ------- */

#define SGNE16(x) ( ((x) & 0x8000) ? - ((x) & 0x7fff) : ((x) & 0x7fff) )
#define SGND16(x) ( ((int)(x) < 0) ? (0x8000 | ((x) & 0x7fff)):((x) & 0x7fff))

#define ISTREAM 1

void CPU::Reset()
{
  memset(m_regs, 0, CPU::NReg * sizeof(word) );
  m_pc = 0;
  m_i = 0;
  m_l = 1;

  m_count_insn = m_count_jmp = 0;
}

bool CPU::Step(int n)
{
  int x;
  for(x = 0; x < n; x++)
    if(!Step())
      return false;
  return true;
}

#define Mwrite(a,d) m_mem->Write((a),(d))
#define Mread(a) m_mem->Read((a))

bool CPU::Step()
{
  word tmp;
  word insn = Mread(m_pc);
  bool alu12;
  word insn_op, insn_ifield, insn_ra, insn_rb;
  word ra, rb;
  bool rel_ra, rel_rb;

  int trace_pc = m_pc, trace_op = (insn & 0xfc00) >> 10,
    trace_areg, trace_a, trace_breg, trace_b,
    trace_dreg, trace_d;

  m_count_insn++;

  if(insn & 0x8000) // JMP
  {
    m_count_jmp++;
    m_pc = insn & 0x7fff;
    trace_areg = 0;
    trace_a = m_pc;
    trace_breg = 0;
    trace_b = 0;
    trace_dreg = 0;
    trace_d = 0;
    goto out;
  }

  // grab the opcode, the I-mode, the regs
  insn_op = (insn & 0xfc00) >> 10;
  insn_ifield = (insn & 0x0300) >> 8;
  insn_ra = (insn & 0x00f0) >> 4;
  insn_rb = (insn & 0x000f);

  rel_ra = (insn_op >= 4 && insn_op < 32) && (insn_op != 22);
  rel_rb = (insn_op >= 8 && insn_op <= 13) || (insn_op == 18) ||
    (insn_op == 20) || (insn_op == 22);

  // handle stack traps
  if(insn_ifield == 1 || insn_ifield == 3)
    if(
      ((((insn_ra + m_i) & 15) > m_i) && rel_ra) ||
      ((((insn_rb + m_i) & 15) > m_i) && rel_rb))
    {
//      printf("underflow: insn is %04x, pc %04x, i %d\n", insn, m_pc, m_i);
      m_regs[15] = m_pc;
      m_pc = 2; // underflow
      trace_dreg = trace_areg = 15;
      trace_d = 2; trace_a = 2;
      goto out;
    }

  if(insn_ifield == 2 && (rel_ra || rel_rb))
    if(m_i == m_l)
    {
//      printf("overflow: insn is %04x\n", insn);
//      printf("ifield is %d, m_i is %d, m_l is %d\n", insn_ifield, m_i, m_l);
      m_regs[15] = m_pc;
      m_pc = 3; // overflow
      trace_dreg = trace_areg = 15;
      trace_d = 3; trace_a = 3;
      goto out;
    }

  // calc the reg indices
  if(insn_ifield == 0)
  {
    ra = insn_ra;
    rb = insn_rb;
  }
  else
  {
    if(rel_ra)
      ra = (insn_ra + m_i) & 15;
    else
      ra = insn_ra;

    if(rel_rb)
      rb = (insn_rb + m_i) & 15;
    else
      rb = insn_rb;
  }

  // ALU is in 12-bit mode if I is used
  alu12 = (insn_ifield != 0);

  // trace data
  trace_pc = m_pc;
  trace_op = insn_op;
  trace_areg = ra;
  trace_a = m_regs[ra];
  trace_breg = rb;
  trace_b = m_regs[rb];
  trace_d = ra;

  // do the op
  switch(insn_op)
  {
  case 0:
  case 1:
  case 2:
  case 3:
    // trap
    m_regs[15] = m_pc;
    m_pc = 1;
    trace_areg = 0; trace_breg = 0; trace_a = 0; trace_b = 0;
    trace_dreg = 15;
    trace_d = m_pc;
    break;

  case 4:
    // skz
    if(m_regs[ra] == 0)
    {
      // skip insn: if ldim, skip its imm word too
      if(Mread(m_pc + 1) & 0xf000 == 0xe000)
        m_pc += 3;
      else
        m_pc += 2;
    }
    else
      m_pc++;

    trace_d = trace_a;

    break;

  case 5:
    // skn
    if((!alu12 && (m_regs[ra] & 0x8000)) || (alu12 && (m_regs[ra] & 0x800)))
    {
      if(Mread(m_pc + 1) & 0xf000 == 0xe000)
        m_pc += 3;
      else
        m_pc += 2;
    }
    else
      m_pc++;

    trace_d = trace_a;

    break;

  case 6:
    // jr
    m_pc = m_regs[ra];

    trace_d = m_pc;

    break;

  case 7:
    // il
    tmp = (m_l & 15) << 4 | (m_i & 15);
    m_l = (m_regs[ra] >> 4) & 15;
    m_i = m_regs[ra] & 15;
    m_regs[ra] = tmp;
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 8:
    // add
    m_regs[ra] += m_regs[rb];
    if(alu12)
      m_regs[ra] &= 0xfff;
    else
      m_regs[ra] &= 0xffff;
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 9:
    // sub
    m_regs[ra] -= m_regs[rb];
    if(alu12)
      m_regs[ra] &= 0xfff;
    else
      m_regs[ra] &= 0xffff;
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 10:
    // and
    m_regs[ra] &= m_regs[rb];
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 11:
    // or
    m_regs[ra] |= m_regs[rb];
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 12:
    // ld
    m_regs[ra] = Mread(m_regs[rb]);
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 13:
    // st
    Mwrite(m_regs[rb], m_regs[ra]);
    m_pc++;
    trace_dreg = 0; trace_d = 0;
    break;

  case 14:
    // ldim
    m_regs[ra] = Mread(m_pc + 1);
    m_pc += 2;
    trace_d = m_regs[ra];
    break;

  case 15:
    // ldpc
    m_regs[ra] = m_pc;
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 16:
    // srl
    m_regs[ra] = (m_regs[ra] >> 1);
    if(alu12)
      m_regs[ra] &= 0xfff;
    else
      m_regs[ra] &= 0xffff;
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 17:
    // sll
    m_regs[ra] = (m_regs[ra] << 1);
    if(alu12)
      m_regs[ra] &= 0xfff;
    else
      m_regs[ra] &= 0xffff;
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 18:
    // xor
    m_regs[ra] = m_regs[ra] ^ m_regs[rb];
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 19:
    // not
    m_regs[ra] = ~m_regs[ra];
    if(alu12)
      m_regs[ra] &= 0xfff;
    else
      m_regs[ra] &= 0xffff;
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 20:
    // mov
    m_regs[ra] = m_regs[rb];
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 21:
    // movb
    m_regs[ra] = m_regs[rb];
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 22:
    // mova
    m_regs[ra] = m_regs[rb];
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 23:
    // high
    m_regs[ra] = m_regs[ra] | 0x7000;
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 24:
    // addi
    m_regs[ra] = m_regs[ra] + insn_rb;
    if(alu12)
      m_regs[ra] &= 0xfff;
    else
      m_regs[ra] &= 0xffff;
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 25:
    // subi
    m_regs[ra] = m_regs[ra] - insn_rb;
    if(alu12)
      m_regs[ra] &= 0xfff;
    else
      m_regs[ra] &= 0xffff;
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 26:
    // andi
    m_regs[ra] = m_regs[ra] & insn_rb;
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 27:
    // ori
    m_regs[ra] = m_regs[ra] | insn_rb;
    if(alu12)
      m_regs[ra] &= 0xfff;
    else
      m_regs[ra] &= 0xffff;
    m_pc++;
    trace_d = m_regs[ra];
    break;

  case 28:
    // finish
    return false;

  default: // just skip
    m_pc++;
  }

  // update I, if necessary
  if((rel_ra || rel_rb) && insn_ifield == 2)
    m_i++;
  if((rel_ra || rel_rb) && insn_ifield == 3)
    m_i--;

out:
  m_pc &= 0xffff;

#if ISTREAM
  printf("%04x, %d, %01x, %04x, %01x, %04x, %01x, %04x\n",
         trace_pc, trace_op, trace_areg, trace_a, trace_breg, trace_b, trace_dreg, trace_d);
#endif


  return true;
}
